//
//  GlassesTabView.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2025/12/11.
//

import SwiftUI
import ViveGlassKit

struct GlassesTabView: CustomTabView {
    typealias DeviceMode = ContentViewModel.DeviceMode

    @ObservedObject var viewModel: ContentViewModel

    var content: AnyView {
        AnyView(
            VStack(spacing: 32) {
                SectionHeader(icon: .resource(.glassesTab), title: "Glasses Status")

                VStack(alignment: .leading, spacing: Layout.spacing) {
                    HStack() {
                        Text("Device Mode:")
                            .font(.appSubheadline)

                        Spacer()

                        Menu {
                            ForEach(DeviceMode.allCases, id: \.self) { mode in
                                Button {
                                    viewModel.deviceMode = mode
                                } label: {
                                    Text(mode.rawValue)
                                        .font(.appBoldBody)
                                }
                            }
                        } label: {
                            HStack {
                                Text(viewModel.deviceMode.rawValue)
                                    .font(.appBoldBody)
                                    .foregroundColor(.primary)
                                Image(systemName: "chevron.down")
                                    .font(.caption)
                                    .foregroundColor(.secondary)
                            }
                            .padding(.horizontal, 12)
                            .padding(.vertical, 6)
                            .background(Color.gray.opacity(0.2))
                            .cornerRadius(8)
                        }
                    }
                    .padding(.bottom, Layout.spacing)

                    HStack() {
                        Text("Connection Status:")
                            .font(.appSubheadline)

                        Spacer()

                        ConnectionStatusText(
                            isConnected: viewModel.status.isConnected
                        )
                    }

                    ToggleButton(
                        isActive: viewModel.status.isConnected,
                        activeText: "Disconnect",
                        inactiveText: "Connect",
                        action: {
                            Task {
                                await viewModel.toggleConnection()
                            }
                        }
                    )
                }
                .padding(.horizontal, Layout.paddingSmall)
            }
            .frame(maxWidth: .infinity, alignment: .top)
        )
    }
}

// MARK: - Connection Status Text
struct ConnectionStatusText: View {
    let isConnected: Bool

    var body: some View {
        Circle()
            .fill(isConnected ? Color.blue : Color.red)
            .frame(width: 12, height: 12)

        Text(isConnected ? "Connected" : "Disconnected")
            .foregroundStyle(isConnected ? Color.blue : Color.red)
            .font(.appBoldBody)
    }
}

#Preview {
    GlassesTabView(viewModel: ContentViewModel())
        .preferredColorScheme(.dark)
}
