//
//  ContentView.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2025/12/10.
//

import SwiftData
import SwiftUI
import ViveGlassSimulator

struct ContentView: View {
    @Environment(\.modelContext) private var modelContext

    // 4. Initialize the VM as a StateObject
    @StateObject private var vm = ContentViewModel()
    @State private var dismissDelegate: ModalDismissDelegate?

    var body: some View {
        ZStack(alignment: .top) {
            TabView {
                GlassesTabView(viewModel: vm)
                    .tabItem {
                        Image(systemName: "eyeglasses")
                        Text("Glasses")
                    }

                ChatTabView(viewModel: vm.chatVM, enabled: vm.status.isConnected)
                    .background(Color(hex: "#2A3649"))
                    .tabItem {
                        Image(systemName: "message")
                        Text("Chat")
                    }

                AudioTabView(viewModel: vm.audioVM, enabled: vm.status.isConnected)
                    .tabItem {
                        Image(systemName: "microphone")
                        Text("Audio")
                    }

                CameraTabView(viewModel: vm.cameraVM, enabled: vm.status.isConnected)
                    .tabItem {
                        Image(systemName: "camera")
                        Text("Camera")
                    }
            }
            .safeAreaInset(edge: .top) {
                Color.clear.frame(height: 60)
            }
            .onAppear { vm.start() }
            .alert("Device Event", isPresented: $vm.showAlert) {
                Button("OK", role: .cancel) {}
            } message: {
                Text(vm.alertMessage)
            }

            // Header view
            VStack(spacing: 0) {
                HStack {
                    Spacer()
                    Text("GlassesConnect")
                        .font(.appTitle)
                        .foregroundColor(.appTextPrimary)

                    Spacer()

                    Button(action: {
                        PresentingControllerResolver.resolve { vc in
                            vm.openSimulator(from: vc)
                            // Attach delegate to the presented simulator VC
                            DispatchQueue.main.async {
                                if let presented = PresentingControllerResolver.topMostPresented(
                                    from: vc
                                ) {
                                    let delegate = ModalDismissDelegate(vm: vm)
                                    presented.presentationController?.delegate = delegate
                                    dismissDelegate = delegate  // retain delegate
                                }
                            }
                        }
                    }) {
                        Image(systemName: "eyeglasses")
                            .font(.title2)
                            .foregroundColor(.white)
                            .frame(width: 60, height: 60)
                            .background(vm.deviceMode == .simulator ? Color.blue : Color.gray)
                            .clipShape(Circle())
                            .shadow(radius: 4)
                    }
                    .disabled(vm.deviceMode != .simulator)
                    .opacity(vm.deviceMode == .simulator ? 1.0 : 0.5)
                }
                .padding(.horizontal, Layout.padding)
                .frame(height: 60)
                .background(Color.appBackground)

                Divider()
                    .padding(.vertical, Layout.padding)
            }
        }
        .disabled(vm.isConnecting)
        .overlay {
            if vm.isConnecting {
                ProgressView()
                    .padding()
                    .background(Color(UIColor.systemBackground).opacity(0.9))
                    .cornerRadius(10)
            }
        }
    }
}

final class ModalDismissDelegate: NSObject, UIAdaptivePresentationControllerDelegate {
    private weak var vm: ContentViewModel?

    init(vm: ContentViewModel) {
        self.vm = vm
    }

    func presentationControllerDidDismiss(_ presentationController: UIPresentationController) {
        vm?.dismissSimulator()
    }
}

// Utility to obtain hosting UIViewController without storing it in the VM
enum PresentingControllerResolver {
    static func resolve(_ onResolve: @escaping (UIViewController) -> Void) {
        let keyWindow = UIApplication.shared.connectedScenes
            .compactMap { $0 as? UIWindowScene }
            .flatMap { $0.windows }
            .first { $0.isKeyWindow }

        if let root = keyWindow?.rootViewController {
            onResolve(root.topMostPresented())
        }
    }

    static func topMostPresented(from base: UIViewController) -> UIViewController? {
        var top = base
        while let presented = top.presentedViewController {
            top = presented
        }
        return top === base ? nil : top
    }
}

private extension UIViewController {
    func topMostPresented() -> UIViewController {
        var top = self
        while let presented = top.presentedViewController {
            top = presented
        }
        return top
    }
}

#Preview {
    ContentView()
        .preferredColorScheme(.dark)
}
