//
//  ChatTabView.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2025/12/11.
//

import SwiftUI

struct ChatTabView: CustomTabView {
    @ObservedObject var viewModel: ChatViewModel
    @FocusState private var ttsFocused: Bool
    let enabled: Bool

    var content: AnyView {
        AnyView(
            VStack {
                headerView
                ttsSection
                sttSection
                Spacer()
            }
            .frame(maxWidth: .infinity, maxHeight: .infinity)
            .contentShape(Rectangle())
            .onTapGesture {
                ttsFocused = false
            }
        )
    }

    private var headerView: some View {
        SectionHeader(icon: .system("ellipsis.message.fill"), title: "Text Capture")
    }

    private var ttsSection: some View {
        let ttsDisable = viewModel.ttsMessage.isEmpty || viewModel.isPlayingTTS || !enabled

        return ActionSection(title: "TTS Input") {
            HStack {
                ZStack(alignment: .topLeading) {
                    RoundedRectangle(cornerRadius: Layout.cornerRadius)
                        .fill(Color(hex: "#2a3343")!)
                        .stroke(Color(hex: "#4b5563")!, lineWidth: 2)
                        .frame(height: 74)

                    if viewModel.ttsMessage.isEmpty && !ttsFocused {
                        Text(
                            viewModel.isPlayingTTS
                                ? "TTS is playing..." : "Enter text for TTS..."
                        )
                        .offset(x: 11, y: 11)
                        .font(.buttonText)
                        .foregroundColor(.appTextTertiary)
                    }

                    TextEditor(text: $viewModel.ttsMessage)
                        .scrollContentBackground(.hidden)
                        .foregroundColor(.white)
                        .tint(.blue)
                        .frame(height: 70)
                        .focused($ttsFocused)
                        .font(.appBody)
                        .padding(8)
                }
                .frame(height: 74)
                .padding(.top, 4)

                Button {
                    Task {
                        await viewModel.sendTTSMessage()
                    }
                    ttsFocused = false
                } label: {
                    Image(systemName: "paperplane.fill")
                        .frame(width: 21, height: 21)
                        .padding(8)
                        .background(
                            ttsDisable ? Color.gray.opacity(0.5) : Color.blue
                        )
                        .cornerRadius(Layout.cornerRadius)
                        .foregroundColor(.white)
                }
                .disabled(ttsDisable)
                .opacity(ttsDisable ? 0.5 : 1.0)
            }
        }
    }

    private var sttSection: some View {
        ActionSection(title: "STT Output") {
            HStack {
                ScrollViewReader { proxy in
                    ScrollView {
                        VStack(spacing: 0) {
                            Text(viewModel.sttMessage)
                                .frame(maxWidth: .infinity, alignment: .leading)
                                .lineLimit(nil)
                                .font(.appBody)
                                .foregroundColor(.appTextTertiary)
                                .padding(8)

                            Color.clear
                                .frame(height: 0)
                                .id("scroll_anchor")
                        }
                    }
                    .frame(height: 74)
                    .background(
                        RoundedRectangle(cornerRadius: Layout.cornerRadius)
                            .fill(Color(hex: "#2a3343")!)
                            .stroke(Color(hex: "#4b5563")!, lineWidth: 2)
                    )
                    .cornerRadius(Layout.cornerRadius)
                    .frame(maxWidth: .infinity, alignment: .leading)
                    .onChange(of: viewModel.sttMessage) {
                        withAnimation(.easeOut(duration: 0.3)) {
                            proxy.scrollTo("scroll_anchor", anchor: .bottom)
                        }
                    }
                }

                micButton
            }
        }
    }

    private var micButton: some View {
        Button {
            Task {
                await viewModel.startRecording()
            }
        } label: {
            Image(
                systemName: viewModel.isRecording
                    ? "microphone.badge.ellipsis.fill" : "mic.fill"
            )
            .frame(width: 21, height: 21)
            .padding(8)
            .background(viewModel.isRecording ? Color.green.opacity(0.5) : Color.green)
            .cornerRadius(8)
            .foregroundColor(.white)
        }
        .accessibilityLabel(
            viewModel.isRecording ? "Stop Recording" : "Start Recording"
        )
        .accessibilityAddTraits(.isButton)
        .disabled(viewModel.isRecording || !enabled)
        .opacity(viewModel.isRecording || !enabled ? 0.5 : 1.0)
    }
}

#Preview {
    struct PreviewContainer: View {
        var body: some View {
            let viewModel = ContentViewModel()
            ChatTabView(viewModel: viewModel.chatVM, enabled: true)
                .background(Color.appBackground)
        }
    }

    return PreviewContainer()
}
