//
//  CameraTabView.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2025/12/11.
//

import SwiftUI

struct CameraTabView: CustomTabView {
    @ObservedObject var viewModel: CameraViewModel
    let enabled: Bool

    var content: AnyView {
        AnyView(
            ZStack {
                VStack {
                    headerView
                    takePhotoView
                    startRecordingView
                    mediaPreviewView
                }
            }
        )
    }

    private var headerView: some View {
        SectionHeader(
            icon: .system("camera.fill"),
            title: "Photo & Video Capture"
        )
    }

    private var takePhotoView: some View {
        VStack(alignment: .leading, spacing: Layout.spacing) {
            HStack(spacing: 8) {
                Image(systemName: "camera.fill")
                    .resizable()
                    .scaledToFit()
                    .frame(width: 20, height: 20)
                    .foregroundColor(.blue)

                Text("Take Photo")
                    .font(.appSubheadline)
                    .foregroundColor(.appTextPrimary)
            }

            Button {
                viewModel.captureImage()
            } label: {
                HStack {
                    Text("Take Photo")
                        .font(.buttonText)
                    if viewModel.isCapturingImage {
                        ProgressView()
                            .progressViewStyle(CircularProgressViewStyle(tint: .white))
                            .scaleEffect(0.8)
                    }
                }
                .frame(maxWidth: .infinity)
                .padding(.vertical, 12)
            }
            .disabled(viewModel.isRecording || viewModel.isCapturingImage || !enabled)
            .buttonStyle(PlainButtonStyle())
            .background(
                viewModel.isRecording || !enabled ? Color.gray.opacity(0.5) : .blue
            )
            .cornerRadius(Layout.cornerRadius)
        }
        .padding(.horizontal, 8)
        .padding(.top, 16)
    }

    private var startRecordingView: some View {
        VStack(alignment: .leading, spacing: Layout.spacing) {
            HStack(spacing: 8) {
                Image(systemName: "video.fill")
                    .resizable()
                    .scaledToFit()
                    .frame(width: 20, height: 20)
                    .foregroundColor(.green)

                Text("Video Recording")
                    .font(.appSubheadline)
                    .foregroundColor(.appTextPrimary)
            }

            ToggleButton(
                isActive: viewModel.isRecording,
                activeText: "Stop Recording",
                inactiveText: "Start Recording",
                activeColor: .red,
                inactiveColor: .green,
                isDisabled: viewModel.isLoadingRecording || !enabled,
                isLoading: viewModel.isLoadingRecording,
                action: {
                    viewModel.toggleRecording()
                }
            )
        }
        .padding(.horizontal, 8)
        .padding(.top, 16)
    }

    private var mediaPreviewView: some View {
        VStack(alignment: .leading, spacing: Layout.spacing) {
            HStack(spacing: 8) {
                Image(systemName: "desktopcomputer")
                    .resizable()
                    .scaledToFit()
                    .frame(width: 20, height: 20)
                    .foregroundColor(.blue)

                Text("Media Preview")
                    .font(.appSubheadline)
                    .foregroundColor(.appTextPrimary)
            }

            ZStack {
                if viewModel.isRecording {
                    VideoLayerView(layer: viewModel.displayLayer)
                        .background(Color.black)
                        .cornerRadius(12)
                        .aspectRatio(16 / 9, contentMode: .fit)
                } else if let image = viewModel.capturedImage {
                    Image(uiImage: image)
                        .resizable()
                        .scaledToFit()
                        .frame(maxWidth: .infinity, maxHeight: .infinity)
                } else {
                    RoundedRectangle(cornerRadius: 12)
                        .stroke(
                            Color.gray.opacity(0.5),
                            style: StrokeStyle(lineWidth: 1.5, dash: [6, 4])
                        )
                        .overlay(
                            Text("Live Camera Feed / Last Capture Preview")
                                .font(.system(size: 14)).italic()
                                .foregroundColor(.gray)
                        )
                }
            }
            .frame(maxWidth: .infinity)

            Spacer(minLength: 0)
        }
        .padding(8)
    }
}

#Preview {
    struct PreviewContainer: View {
        var body: some View {
            let viewModel = ContentViewModel()
            CameraTabView(viewModel: viewModel.cameraVM, enabled: true)
                .background(Color.appBackground)
        }
    }

    return PreviewContainer()
}
