//
//  ContentViewModel.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2025/12/10.
//

import Combine
import CoreMedia
import SwiftUI
import ViveGlassKit
import ViveGlassSimulator

// 1. Create a ViewModel to hold the SDK and Simulator lifecycle
// This prevents the SDK from being recreated when the View redraws
class ContentViewModel: ObservableObject, ViveGlassDelegate {
    enum DeviceMode: String, CaseIterable {
        case viveGlass = "VIVE Glass"
        case simulator = "Simulator"
    }

    private var simulator: GlassConnectSimulator!

    // Published state
    @Published var status: DeviceStatus = .disconnected
    @Published var isConnecting = false
    @Published var showAlert = false
    @Published var alertMessage = ""
    @Published var chatVM = ChatViewModel()
    @Published var audioVM: AudioViewModel!
    @Published var cameraVM = CameraViewModel()
    @Published var deviceMode: DeviceMode = .simulator {
        didSet {
            if oldValue != deviceMode {
                LogUtil.d(TAG, "Update device mode to \(deviceMode.rawValue)")
                ViveGlass.shared.disconnect()
                status = .disconnected
                switchDeviceMode()
                Task {
                    await self.doConnect()
                }
            }
        }
    }

    let TAG = "ContentViewModel"

    init() {
        simulator = GlassConnectSimulator(delegate: self)
        audioVM = AudioViewModel(glass: simulator)
    }

    private func switchDeviceMode() {
        switch deviceMode {
        case .viveGlass:
            ViveGlass.shared.delegate = self
            ViveGlass.shared.setAdapter(nil)
            audioVM = AudioViewModel(glass: ViveGlass.shared)
        case .simulator:
            ViveGlass.shared.setAdapter(simulator)
            ViveGlass.shared.delegate = self
            audioVM = AudioViewModel(glass: simulator)
        }
    }

    // Lifecycle control
    func start() {
        switchDeviceMode()
        Task {
            await self.doConnect()
        }
    }

    // Intent for the View to access simulator
    func openSimulator(from presentingVC: UIViewController) {
        Task { @MainActor in
            await simulator.showSimulatorUI(from: presentingVC)
        }
    }

    func dismissSimulator() {
        simulator.dismissSimulatorUI()
    }

    func doConnect() async {
        guard status.isConnected == false else { return }

        isConnecting = true
        defer { isConnecting = false }

        try? await withTimeout(seconds: 5) {
            do {
                try await ViveGlass.shared.connect()
            } catch {
                LogUtil.e(self.TAG, "Connection failed: \(error)")
                ToastManager.show(message: "Connection failed: \(error.localizedDescription)")
            }
        }
    }

    func toggleConnection() async {
        isConnecting = true
        defer { isConnecting = false }

        try? await withTimeout(seconds: 5) {
            if self.status.isConnected {
                ViveGlass.shared.disconnect()
            } else {
                do {
                    try await ViveGlass.shared.connect()
                } catch {
                    LogUtil.e(self.TAG, "Connection failed: \(error)")
                    ToastManager.show(message: "Connection failed: \(error.localizedDescription)")
                }
            }
        }
    }

    func glasses(
        didUpdateConnectionState state: ViveGlass.ConnectionState
    ) {
        LogUtil.d(TAG, "Connection State Updated: \(state)")
        updateDeviceState(state)
        // TODO
        Task {
            let info = try? await ViveGlass.shared.getUserPreferredLocale()
            print("Glass locale: \(String(describing: info))")
        }
    }

    func glasses(didReceiveCaptureEvent event: ViveGlass.CaptureEvent, imageData: Data) {
        if event == .success, let image = UIImage(data: imageData) {
            DispatchQueue.main.async {
                self.cameraVM.capturedImage = image
            }
        }
    }

    func glasses(didReceiveKeyEvent event: ViveGlass.KeyEvent) {
        // TODO
        LogUtil.d(TAG, "Received Key Event: \(event)")
    }

    func glasses(didReceiveTranscribedEvent event: ViveGlass.TranscribedEvent, text: String?) {
        DispatchQueue.main.async {
            self.chatVM.handleSTTEvent(event, text: text)
        }
    }

    func glasses(didReceiveSynthesisEvent event: ViveGlass.SynthesisEvent) {
        DispatchQueue.main.async {
            self.chatVM.handleTTSEvent(event)
        }
    }

    // MARK: - Helpers
    private func updateDeviceState(
        _ connState: ViveGlass.ConnectionState?
    ) {
        Task { @MainActor in
            self.status = DeviceStatus(
                connectionState: connState ?? self.status.connectionState,
            )

            // Propagate connection state to child VMs
            audioVM.updateConnectionState(connState)
            chatVM.updateConnectionState(connState)
            cameraVM.updateConnectionState(connState)
        }
    }
}
