//
//  ChatViewModel.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Megan Tsai(蔡靚姚) on 2025/12/12.
//

import Combine
import Foundation
import SwiftUI
import ViveGlassKit
import ViveGlassSimulator

class ChatViewModel: ObservableObject {
    @Published var ttsMessage: String = ""
    @Published var isPlayingTTS: Bool = false

    @Published var sttMessage: String =
        "Received message from user... (Click microphone to hear)"
    @Published var isRecording: Bool = false

    private let TIMEOUT_DURATION: TimeInterval = 10

    func sendTTSMessage() async {
        guard !ttsMessage.isEmpty else { return }
        isPlayingTTS = true
        do {
            try await withTimeout(seconds: TIMEOUT_DURATION) {
                let res = try await ViveGlass.shared.speakText(
                    text: self.ttsMessage,
                    locale: Locale(identifier: "en-US")
                )
                if res < 0 {
                    self.isPlayingTTS = false
                    ToastManager.show(message: "Failed to send TTS message, error code: \(res)")
                }
            }
        } catch is TimeoutError {
            isPlayingTTS = false
            ToastManager.show(message: "TTS timed out")
        } catch {
            isPlayingTTS = false
            ToastManager.show(message: "Failed to send TTS message: \(error)")
        }
    }

    func startRecording() async {
        isRecording = true
        sttMessage = "Listening..."
        do {
            try await withTimeout(seconds: TIMEOUT_DURATION) {
                let res = try await ViveGlass.shared.startTranscription(continuous: false)
                if res < 0 {
                    self.isRecording = false
                    ToastManager.show(message: "Failed to start STT, error code: \(res)")
                }
            }
        } catch is TimeoutError {
            isRecording = false
            ToastManager.show(message: "STT timed out")
        } catch {
            self.isRecording = false
            ToastManager.show(message: "Failed to start STT: \(error)")
        }
    }

    func handleSTTEvent(_ event: ViveGlass.TranscribedEvent, text: String?) {
        isRecording = false
        sttMessage = text ?? ""

        let finalEvent: String? =
            switch event {
            case .error:
                "STT Error"
            case .error_resource_conflict:
                "Incall, cannot use STT"
            default:
                nil
            }

        if (finalEvent != nil) {
            ToastManager.show(message: finalEvent ?? "")
        }

    }

    func handleTTSEvent(_ event: ViveGlass.SynthesisEvent) {
        isPlayingTTS = false
        ttsMessage = ""

        let finalEvent: String? =
            switch event {
            case .error:
                "TTS Error"
            case .error_resource_conflict:
                "Incall, cannot use TTS"
            default:
                nil
            }

        if (finalEvent != nil) {
            ToastManager.show(message: finalEvent ?? "")
        }
    }

    @MainActor
    func updateConnectionState(_ connState: ViveGlass.ConnectionState?) {
        if connState == .disconnected || connState == .error {
            isRecording = false
            isPlayingTTS = false
            ttsMessage = ""
        }
    }
}
