//
//  CameraTabViewModel.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Megan Tsai(蔡靚姚) on 2025/12/26.
//

import AVFoundation
import Combine
import CoreMedia
import SwiftUI
import ViveGlassKit
import ViveGlassSimulator

final class CameraViewModel: ObservableObject {
    @Published var capturedImage: UIImage? = nil
    @Published var isCapturingImage: Bool = false
    @Published var isRecording: Bool = false
    @Published var isLoadingRecording: Bool = false
    @Published private(set) var displayLayer: AVSampleBufferDisplayLayer? = nil

    private var pipeline: MediaStreamingManager?

    // MARK: Capture Image
    func captureImage() {
        isCapturingImage = true

        Task(priority: .userInitiated) {
            do {
                _ = try await ViveGlass.shared.captureImage(
                    quality: ViveGlass.ImageQuality.defaultquality
                )
                self.isCapturingImage = false
            } catch {
                self.isCapturingImage = false
                ToastManager.show(message: "Failed to take photo: \(error)")
            }
        }
    }

    // MARK: Recording Control
    func toggleRecording() {
        if isRecording {
            stopRecording()
        } else {
            capturedImage = nil
            startRecording()
        }
    }

    private func startRecording() {
        isRecording = true
        isLoadingRecording = true

        do {
            let p = MediaStreamingManager()
            try p.prepare()
            pipeline = p
            displayLayer = p.displayLayer
        } catch {
            ToastManager.show(message: "Setup pipeline error: \(error)")
            isRecording = false
            isLoadingRecording = false
            return
        }

        Task {
            do {
                _ = try await ViveGlass.shared.startVideoStreaming(
                    onVideoData: { [weak self] buffer in
                        self?.handleVideoFrame(buffer)
                    },
                    onAudioData: { [weak self] buffer in
                        self?.handleAudioFrame(buffer)
                    },
                    onStreamingEvent: { [weak self] event in
                        guard let self else { return }
                        Task { @MainActor in
                            switch event {
                            case .started:
                                self.isRecording = true
                                self.isLoadingRecording = false
                            case .error, .error_resource_conflict, .battery_low,
                                .glasses_overheating:
                                self.isRecording = false
                                self.isLoadingRecording = false
                                ToastManager.show(message: "Video streaming error: \(event)")
                            default:
                                break
                            }
                        }
                    }
                )
            } catch {
                ToastManager.show(message: "Failed to start video streaming: \(error)")
                isRecording = false
                isLoadingRecording = false
            }
        }
    }

    private func stopRecording() {
        self.isRecording = false
        self.isLoadingRecording = false
        ViveGlass.shared.stopVideoStreaming()
        cleanRenderers()
    }

    // MARK: Handle Video
    nonisolated private func handleVideoFrame(_ sampleBuffer: CMSampleBuffer) {
        guard CMSampleBufferDataIsReady(sampleBuffer) else { return }
        let sendableBuffer = UncheckedSendableBuffer(buffer: sampleBuffer)
        DispatchQueue.main.async { [weak self] in
            self?.pipeline?.enqueueVideo(sendableBuffer.buffer)
        }
    }

    // MARK: Handle Audio
    nonisolated private func handleAudioFrame(_ sampleBuffer: CMSampleBuffer) {
        guard CMSampleBufferDataIsReady(sampleBuffer) else { return }
        let sendableBuffer = UncheckedSendableBuffer(buffer: sampleBuffer)
        DispatchQueue.main.async { [weak self] in
            self?.pipeline?.enqueueAudio(sendableBuffer.buffer)
        }
    }

    // MARK: CleanUp
    private func cleanRenderers() {
        pipeline?.stopAndCleanup()
        pipeline = nil
        displayLayer = nil
    }

    @MainActor
    func updateConnectionState(_ connState: ViveGlass.ConnectionState?) {
        if connState == .disconnected || connState == .error {
            if (isRecording) {
                stopRecording()
            }
            capturedImage = nil
        }
    }
}
