//
//  AudioViewModel.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2025/12/29.
//

import AVFoundation
import Combine
import CoreMedia
import ViveGlassKit
import ViveGlassSimulator

class AudioViewModel: ObservableObject {
    private let TAG = "AudioViewModel"

    @Published var isPlaying: Bool = false
    @Published var isRecording: Bool = false
    @Published var isLoadingRecording: Bool = false
    @Published var currentTime: Double?  // In seconds
    @Published var recordedFileURL: URL?

    private let glassInstance: GlassConnect

    private let writerQueue = DispatchQueue(
        label: "com.htc.viveglasskit.audio.writer",
        qos: .userInitiated
    )
    private var aacWriter: AACFileWriter?
    private var recordingURL: URL?
    private var lastRecordedSecond: Int = -1
    private let SAMPLE_RATE = 44100
    private let CHANNEL_COUNT = 2  // Stereo
    private let BIT_RATE = 32000

    init(glass: GlassConnect) {
        self.glassInstance = glass
    }

    private func setupAacWriter(sampleRate: Int, channels: Int) {
        let documentsPath = FileManager.default.urls(for: .documentDirectory, in: .userDomainMask)[
            0
        ]
        let fileName = "recording_\(Date().timeIntervalSince1970).aac"
        let url = documentsPath.appendingPathComponent(fileName)
        recordingURL = url

        do {
            aacWriter = try AACFileWriter(
                outputURL: url,
                sampleRate: sampleRate,
                channels: channels,
                aacProfile: 2
            )
        } catch {
            LogUtil.e(TAG, "Failed to setup AAC writer: \(error)")
        }
    }

    private func closeAacWriter() {
        writerQueue.sync {
            let writer = self.aacWriter
            self.aacWriter = nil
            let finalURL = self.recordingURL
            self.recordingURL = nil

            if let writer {
                writer.close()
            }

            Task { @MainActor in
                self.recordedFileURL = finalURL
            }
        }
    }

    // MARK: - Streaming

    func startAudioStreaming() {
        recordedFileURL = nil
        isLoadingRecording = true
        setupAacWriter(sampleRate: SAMPLE_RATE, channels: CHANNEL_COUNT)

        Task { @MainActor in
            do {
                _ = try await glassInstance.startAudioStreaming(
                    microphone: .towarduser,
                    bitrate: BIT_RATE,
                    sampleRate: SAMPLE_RATE,
                    channelCount: .stereo,
                    onAudioData: { [weak self] sampleBuffer in
                        guard let self = self else { return }

                        let timestamp = CMTimeGetSeconds(
                            CMSampleBufferGetPresentationTimeStamp(sampleBuffer)
                        )
                        let currentSecond = Int(timestamp)
                        let sendableBuffer = UncheckedSendableBuffer(buffer: sampleBuffer)

                        self.writerQueue.async {
                            guard let writer = self.aacWriter else { return }
                            if currentSecond != self.lastRecordedSecond {
                                self.lastRecordedSecond = currentSecond
                                Task { @MainActor in
                                    self.currentTime = Double(currentSecond)
                                }
                            }
                            writer.write(sampleBuffer: sendableBuffer.buffer)
                        }
                    },
                    onStreamingEvent: { [weak self] event in
                        guard let self else { return }
                        Task { @MainActor in
                            switch event {
                            case .started:
                                self.isRecording = true
                                self.isLoadingRecording = false
                                self.playSampleAudio(isFromRecording: true)
                            case .stopped, .error:
                                self.isRecording = false
                                self.isLoadingRecording = false
                                self.currentTime = nil
                                self.stopSampleAudio(isFromRecording: true)
                                self.closeAacWriter()
                            default:
                                break
                            }
                        }
                    }
                )
            } catch {
                ToastManager.show(message: "Failed to start audio streaming: \(error)")
                isRecording = false
                isLoadingRecording = false
            }
        }
    }

    func stopAudioStreaming() {
        glassInstance.stopAudioStreaming()
    }

    func playSampleAudio(isFromRecording: Bool = false) {
        if let simulator = glassInstance as? GlassConnectSimulator {
            simulator.playSampleAudio()
            Task { @MainActor in
                if (isFromRecording) {
                    return
                }
                isPlaying = true
            }
        } else {
            if (!isFromRecording) {
                ToastManager.show(message: "Sample Audio is only available in Simulator mode.")
            }
        }
    }

    func stopSampleAudio(isFromRecording: Bool = false) {
        if let simulator = glassInstance as? GlassConnectSimulator {
            simulator.stopSampleAudio()
            Task { @MainActor in
                if (isFromRecording) {
                    return
                }
                isPlaying = false
            }
        }
    }

    func togglePlaySample() {
        if isPlaying {
            stopSampleAudio()
        } else {
            playSampleAudio()
        }
    }

    func toggleRecording() {
        if isRecording {
            stopAudioStreaming()
        } else {
            startAudioStreaming()
        }
    }

    @MainActor
    func updateConnectionState(_ connState: ViveGlass.ConnectionState?) {
        if connState == .disconnected || connState == .error {
            if (isRecording) {
                isRecording = false
                isLoadingRecording = false
                currentTime = nil
                stopSampleAudio(isFromRecording: true)
                closeAacWriter()
            } else if (isPlaying) {
                stopSampleAudio()
            }
        }
    }
}
