//
//  ToastManager.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2026/2/6.
//

import Combine
import SwiftUI

@MainActor
class ToastManager: ObservableObject {
    static let shared = ToastManager()

    @Published var isShowing: Bool = false
    @Published var message: String = ""

    private var hideWorkItem: DispatchWorkItem?

    private init() {}

    /// Displays a toast message with optional duration
    /// - Parameters:
    ///   - message: The text to display
    ///   - duration: How long to show the toast (default: 2.0 seconds)
    func show(message: String, duration: TimeInterval = 2.0) {
        hideWorkItem?.cancel()
        hideWorkItem = nil

        self.isShowing = false
        self.message = message

        withAnimation(.easeInOut(duration: 0.2)) {
            self.isShowing = true
        }

        let workItem = DispatchWorkItem { [weak self] in
            guard let self = self else { return }
            withAnimation(.easeInOut(duration: 0.2)) {
                self.isShowing = false
                self.message = ""
            }
        }

        hideWorkItem = workItem
        DispatchQueue.main.asyncAfter(deadline: .now() + duration, execute: workItem)
    }
}

extension ToastManager {
    static func show(message: String, duration: TimeInterval = 2.0) {
        shared.show(message: message, duration: duration)
    }
}
