//
//  AudioPlayer.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2026/1/30.
//

import AVFoundation
import CoreMedia

class AudioPlayer {
    private let TAG = "AudioPlayer"

    private var audioEngine: AVAudioEngine?
    private var playerNode: AVAudioPlayerNode?
    private var audioFormat: AVAudioFormat?
    private var aacConverter: AVAudioConverter?
    private var resamplingConverter: AVAudioConverter?
    private var lastInputSampleRate: Double?

    func start() {
        do {
            let audioSession = AVAudioSession.sharedInstance()
            try audioSession.setCategory(.playback, mode: .default)
            try audioSession.setActive(true)

            let engine = AVAudioEngine()
            let player = AVAudioPlayerNode()

            engine.attach(player)

            // Use hardware output format for best compatibility
            let outputFormat = engine.mainMixerNode.outputFormat(forBus: 0)

            engine.connect(player, to: engine.mainMixerNode, format: outputFormat)

            try engine.start()
            player.play()

            self.audioEngine = engine
            self.playerNode = player
            self.audioFormat = outputFormat
        } catch {
            LogUtil.e(TAG, "Failed to start audio engine: \(error)")
        }
    }

    func stop() {
        playerNode?.stop()
        audioEngine?.stop()
        playerNode = nil
        audioEngine = nil
        aacConverter = nil
        resamplingConverter = nil
        lastInputSampleRate = nil
    }

    func play(sampleBuffer: CMSampleBuffer) {
        guard let player = playerNode,
            let outputFormat = audioFormat
        else { return }

        guard
            let pcmBuffer = convertToPCMBuffer(
                sampleBuffer: sampleBuffer,
                outputFormat: outputFormat
            )
        else {
            return
        }

        player.scheduleBuffer(pcmBuffer, at: nil, options: [], completionHandler: nil)
    }

    private func convertToPCMBuffer(sampleBuffer: CMSampleBuffer, outputFormat: AVAudioFormat)
        -> AVAudioPCMBuffer?
    {
        guard let formatDesc = CMSampleBufferGetFormatDescription(sampleBuffer),
            let asbd = CMAudioFormatDescriptionGetStreamBasicDescription(formatDesc)
        else {
            return nil
        }

        guard let inputFormat = AVAudioFormat(streamDescription: asbd) else {
            return nil
        }

        if asbd.pointee.mFormatID == kAudioFormatMPEG4AAC {
            return decodeAACToPCM(
                sampleBuffer: sampleBuffer,
                inputFormat: inputFormat,
                outputFormat: outputFormat
            )
        }

        // Handle uncompressed audio
        var blockBuffer: CMBlockBuffer?
        var audioBufferList = AudioBufferList()

        let status = CMSampleBufferGetAudioBufferListWithRetainedBlockBuffer(
            sampleBuffer,
            bufferListSizeNeededOut: nil,
            bufferListOut: &audioBufferList,
            bufferListSize: MemoryLayout<AudioBufferList>.size,
            blockBufferAllocator: nil,
            blockBufferMemoryAllocator: nil,
            flags: 0,
            blockBufferOut: &blockBuffer
        )

        guard status == noErr else { return nil }

        let frameCount = CMSampleBufferGetNumSamples(sampleBuffer)
        guard
            let pcmBuffer = AVAudioPCMBuffer(
                pcmFormat: inputFormat,
                frameCapacity: AVAudioFrameCount(frameCount)
            )
        else {
            return nil
        }

        pcmBuffer.frameLength = AVAudioFrameCount(frameCount)
        memcpy(
            pcmBuffer.audioBufferList.pointee.mBuffers.mData,
            audioBufferList.mBuffers.mData,
            Int(audioBufferList.mBuffers.mDataByteSize)
        )

        return pcmBuffer
    }

    private func decodeAACToPCM(
        sampleBuffer: CMSampleBuffer,
        inputFormat: AVAudioFormat,
        outputFormat: AVAudioFormat
    ) -> AVAudioPCMBuffer? {
        // Recreate converters if input sample rate changed
        if lastInputSampleRate != inputFormat.sampleRate {
            lastInputSampleRate = inputFormat.sampleRate
            aacConverter = nil
            resamplingConverter = nil

            // Intermediate format at input sample rate
            guard
                let intermediateFormat = AVAudioFormat(
                    standardFormatWithSampleRate: inputFormat.sampleRate,
                    channels: outputFormat.channelCount
                )
            else {
                LogUtil.e(TAG, "Failed to create intermediate format")
                return nil
            }

            aacConverter = AVAudioConverter(from: inputFormat, to: intermediateFormat)

            // Create resampler if sample rates differ
            if inputFormat.sampleRate != outputFormat.sampleRate {
                resamplingConverter = AVAudioConverter(from: intermediateFormat, to: outputFormat)
            }
        }

        guard let converter = aacConverter else {
            LogUtil.e(TAG, "No AAC converter available")
            return nil
        }

        guard let blockBuffer = CMSampleBufferGetDataBuffer(sampleBuffer) else {
            return nil
        }

        var totalLength = 0
        var dataPointer: UnsafeMutablePointer<Int8>?
        let dataStatus = CMBlockBufferGetDataPointer(
            blockBuffer,
            atOffset: 0,
            lengthAtOffsetOut: nil,
            totalLengthOut: &totalLength,
            dataPointerOut: &dataPointer
        )

        guard dataStatus == noErr, let data = dataPointer, totalLength > 0 else {
            return nil
        }

        let (aacData, aacLength) = stripADTSHeader(data: data, length: totalLength)
        guard aacLength > 0 else { return nil }

        // Decode AAC to intermediate PCM
        let frameCapacity: AVAudioFrameCount = 4096
        guard
            let decodedBuffer = AVAudioPCMBuffer(
                pcmFormat: converter.outputFormat,
                frameCapacity: frameCapacity
            )
        else {
            return nil
        }

        var error: NSError?
        var inputConsumed = false

        let convertStatus = converter.convert(to: decodedBuffer, error: &error) { _, outStatus in
            if inputConsumed {
                outStatus.pointee = .noDataNow
                return nil
            }
            inputConsumed = true
            outStatus.pointee = .haveData

            let compressedBuffer = AVAudioCompressedBuffer(
                format: inputFormat,
                packetCapacity: 1,
                maximumPacketSize: aacLength
            )

            memcpy(compressedBuffer.data, aacData, aacLength)
            compressedBuffer.byteLength = UInt32(aacLength)
            compressedBuffer.packetCount = 1

            if let packetDescriptions = compressedBuffer.packetDescriptions {
                packetDescriptions[0].mStartOffset = 0
                packetDescriptions[0].mDataByteSize = UInt32(aacLength)
                packetDescriptions[0].mVariableFramesInPacket = 0
            }

            return compressedBuffer
        }

        if error != nil || convertStatus == .error || decodedBuffer.frameLength == 0 {
            return nil
        }

        // Resample if needed
        guard let resampler = resamplingConverter else {
            return decodedBuffer
        }

        guard
            let resampledBuffer = AVAudioPCMBuffer(
                pcmFormat: outputFormat,
                frameCapacity: frameCapacity
            )
        else {
            return nil
        }

        var resampleError: NSError?
        var resampleInputConsumed = false

        resampler.convert(to: resampledBuffer, error: &resampleError) { _, outStatus in
            if resampleInputConsumed {
                outStatus.pointee = .noDataNow
                return nil
            }
            resampleInputConsumed = true
            outStatus.pointee = .haveData
            return decodedBuffer
        }

        return resampledBuffer.frameLength > 0 ? resampledBuffer : nil
    }

    private func stripADTSHeader(data: UnsafeMutablePointer<Int8>, length: Int) -> (
        UnsafePointer<Int8>, Int
    ) {
        let byte0 = UInt8(bitPattern: data[0])
        let byte1 = UInt8(bitPattern: data[1])

        let hasADTSHeader = (byte0 == 0xFF) && ((byte1 & 0xF0) == 0xF0)

        guard hasADTSHeader, length >= 7 else {
            return (UnsafePointer(data), length)
        }

        let protectionAbsent = (byte1 & 0x01) == 1
        let headerLength = protectionAbsent ? 7 : 9

        let byte3 = UInt8(bitPattern: data[3])
        let byte4 = UInt8(bitPattern: data[4])
        let byte5 = UInt8(bitPattern: data[5])

        let frameLength = (Int(byte3 & 0x03) << 11) | (Int(byte4) << 3) | (Int(byte5 & 0xE0) >> 5)

        guard frameLength <= length, frameLength > headerLength else {
            return (UnsafePointer(data), length)
        }

        let aacDataPointer = data.advanced(by: headerLength)
        let aacLength = frameLength - headerLength

        return (UnsafePointer(aacDataPointer), aacLength)
    }
}
