//
//  AsyncUtils.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2026/2/9.
//

import Foundation

/// Executes an async operation with a timeout
/// - Parameters:
///   - seconds: Maximum time to wait before throwing TimeoutError
///   - operation: The async operation to execute
/// - Returns: The result of the operation if completed within timeout
/// - Throws: TimeoutError if operation exceeds timeout, or any error thrown by operation
func withTimeout<T>(
    seconds: TimeInterval,
    operation: @escaping () async throws -> T
) async throws -> T {
    try await withThrowingTaskGroup(of: T.self) { group in
        group.addTask {
            try await operation()
        }

        group.addTask {
            try await Task.sleep(nanoseconds: UInt64(seconds * 1_000_000_000))
            throw TimeoutError()
        }

        let result = try await group.next()!
        group.cancelAll()
        return result
    }
}

/// Error thrown when an async operation exceeds its timeout
struct TimeoutError: Error, LocalizedError {
    var errorDescription: String? {
        "Operation timed out"
    }
}
