//
//  AacWriter.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Created by Hank Chiu on 2025/12/30.
//

import AVFoundation
import AudioToolbox

final class AACFileWriter {
    private let TAG = "AACFileWriter"

    private var fileHandle: FileHandle?
    private let path: URL
    private let sampleRate: Int
    private let channels: Int
    private let aacProfile: Int  // 2 = AAC LC
    private var aacObjectTypeIndex: Int  // ADTS profile index (0: Main, 1: LC, 2: SSR, 3: LTP)

    init(outputURL: URL, sampleRate: Int, channels: Int, aacProfile: Int = 2) throws {
        self.path = outputURL
        self.sampleRate = sampleRate
        self.channels = channels
        self.aacProfile = aacProfile
        self.aacObjectTypeIndex = max(0, min(3, aacProfile - 1))  // LC(2) -> 1

        try FileManager.default.createDirectory(
            at: outputURL.deletingLastPathComponent(),
            withIntermediateDirectories: true
        )
        FileManager.default.createFile(atPath: outputURL.path, contents: nil)
        guard let handle = try? FileHandle(forWritingTo: outputURL) else {
            throw NSError(
                domain: "AACFileWriter",
                code: -1,
                userInfo: [NSLocalizedDescriptionKey: "Cannot open output file"]
            )
        }
        self.fileHandle = handle
    }

    func close() {
        try? fileHandle?.close()
        fileHandle = nil
    }

    func write(sampleBuffer: CMSampleBuffer) {
        guard let handle = fileHandle else { return }
        guard let block = CMSampleBufferGetDataBuffer(sampleBuffer) else {
            LogUtil.e(TAG, "Failed to get data buffer from sample buffer")
            return
        }

        // Read payload bytes
        let length = CMBlockBufferGetDataLength(block)
        if length <= 0 {
            LogUtil.e(TAG, "Block buffer length is zero")
            return
        }

        var buffer = [UInt8](repeating: 0, count: length)
        let status = CMBlockBufferCopyDataBytes(
            block,
            atOffset: 0,
            dataLength: length,
            destination: &buffer
        )
        if status != kCMBlockBufferNoErr {
            LogUtil.e(TAG, "Failed to copy data bytes from block buffer, status: \(status)")
            return
        }

        // ADTS header + payload
        let adtsHeader = makeADTSHeader(aacFrameLength: buffer.count)
        handle.write(Data(adtsHeader))
        handle.write(Data(buffer))
    }

    private func samplingFrequencyIndex(for rate: Int) -> Int {
        let table: [Int: Int] = [
            96000: 0, 88200: 1, 64000: 2, 48000: 3, 44100: 4, 32000: 5,
            24000: 6, 22050: 7, 16000: 8, 12000: 9, 11025: 10, 8000: 11, 7350: 12,
        ]
        return table[rate] ?? 3
    }

    private func makeADTSHeader(aacFrameLength: Int) -> [UInt8] {
        let sfIndex = samplingFrequencyIndex(for: sampleRate)
        let channelConfig = channels
        let adtsLen = 7
        let fullLength = adtsLen + aacFrameLength

        var header = [UInt8](repeating: 0, count: adtsLen)
        header[0] = 0xFF
        header[1] = 0xF1

        // Break up header[2] into sub-expressions to help type-checking
        let profileBits: UInt8 = UInt8((aacObjectTypeIndex & 0x3) << 6)
        let sfBits: UInt8 = UInt8((sfIndex & 0xF) << 2)
        let chMsbBit: UInt8 = UInt8((channelConfig >> 2) & 0x1)
        header[2] = profileBits | sfBits | chMsbBit

        // Also split header[3]..[5] to avoid type-check issues
        let chLsbBits: UInt8 = UInt8((channelConfig & 0x3) << 6)
        let lenHighBits: UInt8 = UInt8((fullLength >> 11) & 0x03)
        header[3] = chLsbBits | lenHighBits

        header[4] = UInt8((fullLength >> 3) & 0xFF)

        let lenLowBits: UInt8 = UInt8(fullLength & 0x7)
        header[5] = UInt8((lenLowBits << 5) | 0x1F)

        header[6] = 0xFC
        return header
    }
}
