//
//  MediaStreamingPipeline.swift
//  ViveGlassesConnectSDK_ios_samples
//
//  Orchestrates AVSampleBufferRenderSynchronizer, video/audio renderers,
//  and gates playback until first audio+video frames.
//

import AVFoundation
import CoreMedia

final class MediaStreamingManager {
    struct Config {
        // Startup buffer requirements - reduced for faster startup
        var minStartupAudioBuffers: Int = 3
        var minStartupVideoFrames: Int = 1
    }

    private(set) var displayLayer: AVSampleBufferDisplayLayer?

    private var synchronizer: AVSampleBufferRenderSynchronizer?
    private var videoRenderer: VideoRenderer?
    private var audioRenderer: AudioRenderer?

    private let queue = DispatchQueue(label: "com.htc.viveglasses.media.streaming")
    private let queueKey = DispatchSpecificKey<Void>()

    private var didStartSynchronizer = false
    private var gotFirstAudio = false
    private var gotFirstVideo = false
    private var pendingAudio: [CMSampleBuffer] = []
    private var pendingVideo: [CMSampleBuffer] = []

    var config = Config()

    private func withLock(_ work: () -> Void) {
        if DispatchQueue.getSpecific(key: queueKey) != nil {
            work()
        } else {
            queue.sync(execute: work)
        }
    }

    func prepare() throws {
        if synchronizer != nil || videoRenderer != nil || audioRenderer != nil {
            print("[Media] Warning: prepare() called with existing resources, cleaning up first")
            stopAndCleanup()
        }

        queue.setSpecific(key: queueKey, value: ())

        let sync = AVSampleBufferRenderSynchronizer()
        sync.setRate(0.0, time: .zero)
        synchronizer = sync

        let vRender = VideoRenderer()
        vRender.attach(to: sync)
        videoRenderer = vRender
        displayLayer = vRender.layer

        let aRender = AudioRenderer()
        try aRender.startSession()
        aRender.attach(to: sync)
        audioRenderer = aRender

        withLock {
            self.didStartSynchronizer = false
            self.gotFirstAudio = false
            self.gotFirstVideo = false
            self.pendingAudio.removeAll()
            self.pendingVideo.removeAll()
        }
    }

    // MARK: - Enqueue
    func enqueueVideo(_ sampleBuffer: CMSampleBuffer) {
        guard CMSampleBufferDataIsReady(sampleBuffer) else { return }
        queue.async {
            guard let videoRenderer = self.videoRenderer else { return }
            if self.didStartSynchronizer {
                videoRenderer.enqueue(sampleBuffer)
                return
            }
            self.pendingVideo.append(sampleBuffer)
            self.gotFirstVideo = true
            self.tryStartSynchronizerLocked()
        }
    }

    func enqueueAudio(_ sampleBuffer: CMSampleBuffer) {
        guard CMSampleBufferDataIsReady(sampleBuffer) else { return }
        queue.async {
            guard let audioRenderer = self.audioRenderer else { return }
            if self.didStartSynchronizer {
                audioRenderer.enqueue(sampleBuffer)
                return
            }
            self.pendingAudio.append(sampleBuffer)
            self.gotFirstAudio = true
            self.tryStartSynchronizerLocked()
        }
    }

    // MARK: - Start gating
    private func tryStartSynchronizerLocked() {
        guard !didStartSynchronizer else { return }
        guard gotFirstAudio, gotFirstVideo else { return }
        guard let sync = synchronizer,
            let audioRenderer = audioRenderer,
            let videoRenderer = videoRenderer,
            pendingAudio.count >= config.minStartupAudioBuffers,
            pendingVideo.count >= config.minStartupVideoFrames
        else { return }

        let a0 = CMSampleBufferGetPresentationTimeStamp(pendingAudio[0])
        let startPTS = a0

        for a in pendingAudio {
            audioRenderer.enqueue(a)
        }
        for v in pendingVideo {
            videoRenderer.enqueue(v)
        }

        pendingAudio.removeAll()
        pendingVideo.removeAll()

        sync.setRate(1.0, time: startPTS)
        didStartSynchronizer = true
    }

    // MARK: - Cleanup
    func stopAndCleanup() {
        queue.sync {
            synchronizer?.setRate(0.0, time: .zero)
            videoRenderer?.flush()
            audioRenderer?.flush()

            if let sync = synchronizer {
                if let v = videoRenderer { v.detach(from: sync) }
                if let a = audioRenderer { a.detach(from: sync) }
            }

            displayLayer = nil
            videoRenderer = nil
            synchronizer = nil
            audioRenderer?.stopSession()
            audioRenderer = nil

            didStartSynchronizer = false
            gotFirstAudio = false
            gotFirstVideo = false
            pendingAudio.removeAll()
            pendingVideo.removeAll()
        }
    }
}
