package com.htc.viveglass.viveglasssample

import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Matrix
import android.media.AudioManager
import android.media.ExifInterface
import android.media.ExifInterface.ORIENTATION_NORMAL
import android.media.ExifInterface.ORIENTATION_ROTATE_180
import android.media.ExifInterface.ORIENTATION_ROTATE_270
import android.media.ExifInterface.ORIENTATION_ROTATE_90
import android.media.MediaCodec
import android.media.MediaPlayer
import android.view.Surface
import com.htc.viveglass.viveglasssample.util.AudioDecoder
import com.htc.viveglass.viveglasssample.util.H264Decoder
import com.htc.viveglass.viveglasssample.util.StreamingPlayer
import com.htc.viveglass.sdk.simulator.ViveGlassSimulator
import com.htc.viveglass.sdk.AudioChannel
import com.htc.viveglass.sdk.CaptureEvent
import com.htc.viveglass.sdk.ConnectionState
import com.htc.viveglass.sdk.ImagePayload
import com.htc.viveglass.sdk.ImageQuality
import com.htc.viveglass.sdk.KeyEvent
import com.htc.viveglass.sdk.Microphone
import com.htc.viveglass.sdk.StreamingEvent
import com.htc.viveglass.sdk.SynthesisEvent
import com.htc.viveglass.sdk.TranscribedEvent
import com.htc.viveglass.sdk.VideoQuality
import com.htc.viveglass.sdk.ViveGlass
import com.htc.viveglass.sdk.ViveGlassKit
import com.htc.viveglass.sdk.StreamingEvent.*
import com.htc.viveglass.sdk.client.StreamingBufferCallback
import com.htc.viveglass.sdk.client.StreamingEventCallback
import com.htc.viveglass.sdk.client.ViveGlassClientCallback
import com.htc.viveglass.viveglasssample.ui.tab.AppDestination
import com.htc.viveglass.viveglasssample.util.Logger
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import java.io.ByteArrayInputStream
import java.nio.ByteBuffer
import java.util.Locale

interface ViveGlassKitInterface{

    val connection: StateFlow<Boolean>
    val isSampleAudioPlaying : StateFlow<Boolean>
    val isAudioRecording : StateFlow<Boolean>
    val isVideoStreaming : StateFlow<Boolean>
    val isImageCapturing : StateFlow<Boolean>
    val previewRatio : StateFlow<Float>
    val isSimulator : StateFlow<Boolean>
    val isStartTranscribe : StateFlow<Boolean>
    val imageReceived : SharedFlow<Bitmap>
    val keyEvent : SharedFlow<KeyEvent>
    val textTranscribed : StateFlow<String>
    fun setSimulator(isUseSimulator: Boolean)
    fun connect()
    fun disconnect()
    fun isConnected(): Boolean
    fun speakText(text:String)
    fun startTranscription()
    fun stopTranscription()
    fun playSampleAudio()
    fun stopSampleAudio()
    fun startAudioStreaming()
    fun stopAudioStreaming()
    fun captureImage()
    fun startVideoStreaming()
    fun stopVideoStreaming()
    fun getPreferredLocale() : Locale
    fun cleanup()
}

const val TAG:String = "ViveGlassKit"
class ViveGlassKitManager(
    private val glass : ViveGlass,
    private val kit: ViveGlassKit,
    private val simulator: ViveGlassSimulator,
    appContext : Context,
    audioManager :AudioManager
): ViveGlassKitInterface
{
    private val log = Logger.instance
    private val _isSimulator =  MutableStateFlow(false)
    override val isSimulator : StateFlow<Boolean> = _isSimulator.asStateFlow()

    private val _connection = MutableStateFlow(false)
    override val connection: StateFlow<Boolean> = _connection.asStateFlow()

    private val _isSampleAudioPlaying = MutableStateFlow(false)
    override val  isSampleAudioPlaying : StateFlow<Boolean> = _isSampleAudioPlaying.asStateFlow()

    private val _isAudioRecording = MutableStateFlow(false)
    override val  isAudioRecording : StateFlow<Boolean> = _isAudioRecording.asStateFlow()

    private val _isVideoStreaming = MutableStateFlow(false)
    override val isVideoStreaming : StateFlow<Boolean> = _isVideoStreaming.asStateFlow()

    private val _isImageCapturing = MutableStateFlow(false)
    override val isImageCapturing: StateFlow<Boolean> = _isImageCapturing.asStateFlow()

    private val _previewRatio = MutableStateFlow(1.8f)
    override val  previewRatio : StateFlow<Float> = _previewRatio.asStateFlow()

    private val _isStartTranscribe = MutableStateFlow(false)
    override val isStartTranscribe: StateFlow<Boolean> = _isStartTranscribe.asStateFlow()
    private val _textTranscribed = MutableStateFlow("")
    override val textTranscribed: StateFlow<String> = _textTranscribed.asStateFlow()

    private val _imageReceived =MutableSharedFlow<Bitmap>(
        replay = 0,
        extraBufferCapacity = 1,
        onBufferOverflow = BufferOverflow.DROP_OLDEST
    )
    override val imageReceived: SharedFlow<Bitmap> = _imageReceived

    private val _keyEvent = MutableSharedFlow<KeyEvent>(
        replay = 0,
        extraBufferCapacity = 1,
        onBufferOverflow = BufferOverflow.DROP_OLDEST
    )
    override val keyEvent: SharedFlow<KeyEvent> = _keyEvent

    private var mediaPlayer: MediaPlayer =  MediaPlayer.create(appContext, R.raw.default_audio_256kbps)

    private var streamPlayer : StreamingPlayer? = null
    @Volatile private var renderSurface: Surface? = null
    private var videoDecoder: H264Decoder? = null
    private var audioDecoder = AudioDecoder(audioManager)

    // ======================
    // Callbacks
    // ======================

    private val viveGlassClientCallback = object : ViveGlassClientCallback{
        override fun onConnectionStateChanged(state: ConnectionState?) {
            log.d(TAG, "onConnectionStateChanged() state: [$state]")
            when(state)
            {
                ConnectionState.CONNECTING->
                    log.d(TAG, "onConnectionStateChanged() connecting...")
                ConnectionState.CONNECTED ->
                    onConnected()
                ConnectionState.DISCONNECTED ->
                    onDisconnected()
                ConnectionState.ERROR, null ->
                    log.e(TAG, "onConnectionStateChanged() state: [$state]")
            }
        }

        override fun onImageCaptured(
            event: CaptureEvent?,
            payload: ImagePayload?
        ) {
            log.d(TAG, "onImageCaptured() event: [$event]")
            when(event)
            {
                CaptureEvent.SUCCESS -> {
                    if(payload == null) return
                    val bitmap = byteToBitmap(payload.byteArray)
                    _previewRatio.value = (bitmap.width *1.0f) / (bitmap.height*1.0f)
                    _imageReceived.tryEmit(bitmap)
                }
                CaptureEvent.ERROR,
                CaptureEvent.ERROR_RESOURCE_CONFLICT,
                null-> {
                    log.e(TAG, "onImageCaptured() event: [$event]")
                }
            }
            _isImageCapturing.value = false

        }
        override fun onSpeechTranscribed(
            event: TranscribedEvent?,
            text: String?
        ) {
            log.d(TAG, "onSpeechTranscribed() event: [$event], text: [$text]")
            when(event)
            {
                TranscribedEvent.SUCCESS -> {
                    val transcribed = text ?: return
                    _textTranscribed.value = transcribed
                }
                TranscribedEvent.ERROR,
                TranscribedEvent.ERROR_RESOURCE_CONFLICT ->
                {
                    log.e(TAG, "onSpeechTranscribed() event: [$event]")
                }
                null -> log.e(TAG, "onSpeechTranscribed() event: [null]")
            }
            _isStartTranscribe.value = false
        }
        override fun onTextSpoken(event: SynthesisEvent?) {
            when(event)
            {
                SynthesisEvent.SUCCESS ->
                    log.d(TAG, "onTextSpoken() event: [$event]")
                SynthesisEvent.ERROR,
                SynthesisEvent.ERROR_RESOURCE_CONFLICT,
                null ->
                    log.e(TAG, "onTextSpoken() event: [$event]")
                SynthesisEvent.ERROR_UNSUPPORTED_LOCALE ->
                    log.e(TAG, "onTextSpoken() event: [$event]")
            }
        }

        override fun onKeyEvent(event: KeyEvent?) {
            if(event == null) return
            log.d(TAG, "onKeyEvent() event: [$event]")
            _keyEvent.tryEmit(event)
        }
    }

    //--------- Audio Recording Callback ---------------
    private val audioRecordCallback = object : StreamingBufferCallback{
        override fun onReceiveBuffer(
            byteBuffer: ByteBuffer?,
            bufferInfo: MediaCodec.BufferInfo?
        ) {
            if(byteBuffer == null || bufferInfo ==null) return
            if(_isAudioRecording.value) {
                audioDecoder.onReceivedAudioBuffer(byteBuffer, bufferInfo)
            }
        }
    }

    private val audioRecordEventCallback = StreamingEventCallback { event ->
        when(event)
        {
            STARTED -> {
                log.d(TAG, "audioRecordEventCallback() event = [$event]")
                _isAudioRecording.value =true
                audioDecoder.start()
                audioDecoder.setPlaybackEnabled(true)
            }
            STOPPED -> {
                log.d(TAG, "audioRecordEventCallback() event = [$event]")
                _isAudioRecording.value = false
                audioDecoder.stop()
                audioDecoder.setPlaybackEnabled(false)
            }
            ERROR_RESOURCE_CONFLICT,
            GLASS_OVERHEATING ,
            BATTERY_LOW,
            ERROR
                -> log.e(TAG, "audioRecordEventCallback() event = [$event]")

        }
    }

    //--------- Video Recording Callback ---------------
    private val streamEventCallback = object : StreamingEventCallback{
        override fun onStreamingEvent(event: StreamingEvent?) {
            val event = event?:return
            log.d(TAG, "streamEventCallback() event = [$event]")
            when(event)
            {
                STARTED ->
                {
                    _isVideoStreaming.value = true
                    streamPlayer?.onStreamEvent(event)
                }
                STOPPED ->
                {
                    _isVideoStreaming.value = false
                    streamPlayer?.onStreamEvent(event)
                }
                ERROR_RESOURCE_CONFLICT,
                GLASS_OVERHEATING,
                BATTERY_LOW,
                ERROR -> {
                    log.e(TAG, "streamEventCallback() error = [$event]")
                }
            }
        }
    }

    private val audioStreamCallback = object : StreamingBufferCallback{
        override fun onReceiveBuffer(
            byteBuffer: ByteBuffer?,
            bufferInfo: MediaCodec.BufferInfo?
        ) {
            if(byteBuffer == null || bufferInfo ==null) return
            streamPlayer?.onReceivedAudioBuffer(byteBuffer,bufferInfo)
        }
    }

    private val videoStreamCallback = object :StreamingBufferCallback{
        override fun onReceiveBuffer(
            byteBuffer: ByteBuffer?,
            bufferInfo: MediaCodec.BufferInfo?
        ) {
            if(byteBuffer == null || bufferInfo ==null) return
            streamPlayer?.onReceivedVideoBuffer(byteBuffer, bufferInfo)
        }
    }

    private val playerCallback = object : StreamingPlayer.StreamingPlayerCallback {
        override fun onPlaybackEvent(event: StreamingPlayer.PlaybackEvent) {
            when(event)
            {
                StreamingPlayer.PlaybackEvent.EndOfStream ->
                {
                    _isVideoStreaming.value = false
                    log.d(TAG, "onPlaybackEvent() EOS")
                }
                is StreamingPlayer.PlaybackEvent.Error -> TODO()
                StreamingPlayer.PlaybackEvent.Started ->
                {
                    log.d(TAG, "onPlaybackEvent() Started...")
                    val player = streamPlayer ?: return
                    _previewRatio.value = player.getFrameRatio()
                }
                StreamingPlayer.PlaybackEvent.Stopped -> TODO()
            }
        }
    }

    // ==============================
    // Decoder / StreamingPlayer
    // ==============================


    fun attachPreviewSurface(surface: Surface)
    {
        if(renderSurface != null && renderSurface !== surface)
            videoDecoder?.stop()

        renderSurface = surface

        log.d(TAG, "attachPreviewSurface() Surface attached.")
        createVideoDecoder()
    }

    fun detachAndStopPreview() {
        if(renderSurface != null )
        {
            stopVideoStreaming()
        }
        log.d(TAG, "detachAndStopPreview() Surface detached.")
    }

    private fun createVideoDecoder()
    {
        val surf = renderSurface ?: return

        videoDecoder = H264Decoder(surf)
        createStreamPlayer()
    }

    private fun createStreamPlayer()
    {
        val v = videoDecoder ?: return
        val player = StreamingPlayer(v, audioDecoder,callbacks= playerCallback)
        streamPlayer =player

        log.d(TAG,"Decoder created and started")
    }

    // =======================
    // ViveGlassKit
    // =======================

    override fun connect() {
        log.d(TAG, "connect()")
        glass.connect(viveGlassClientCallback)
    }

    override fun disconnect() {
        log.d(TAG, "disconnect()")
        glass.disconnect()
    }

    private fun onConnected(){
        _connection.value = true
    }

    private fun onDisconnected(){
        _connection.value = false
        releasePreviousPage(AppDestination.Chat.route)
        releasePreviousPage(AppDestination.Audio.route)
        releasePreviousPage(AppDestination.Camera.route)
    }

    override fun isConnected(): Boolean {
        return  glass.isConnected
    }

    override fun setSimulator(isUseSimulator: Boolean) {
        if(_isSimulator.value == isUseSimulator && ViveGlass.adapter != null) return
        if(glass.isConnected) glass.disconnect()

        _isSimulator.value = isUseSimulator
        if(isUseSimulator)
            ViveGlass.adapter = simulator
        else
            ViveGlass.adapter = kit
    }

    override fun speakText(text: String) {
        glass.speakText(text, getPreferredLocale())
    }

    override fun startTranscription() {
        glass.startTranscription(false)
        _isStartTranscribe.value = true
    }

    override fun stopTranscription() {
        glass.stopTranscription()
        _isStartTranscribe.value = false
    }

    override fun playSampleAudio() {

        mediaPlayer.setOnCompletionListener {
            _isSampleAudioPlaying.value = false
        }

        if(!mediaPlayer.isPlaying)
        {
            mediaPlayer.start()
            _isSampleAudioPlaying.value = true
        }
    }

    override fun stopSampleAudio() {
        if(mediaPlayer.isPlaying)
        {
            mediaPlayer.pause()
            mediaPlayer.seekTo(0)
            _isSampleAudioPlaying.value = false
        }
    }

    /** startAudioStreaming()
     * Replacing Simulator Audio Samples
     * Path: Transfer your file to the app files directory. Ex:
     * /data/data/com.htc.viveglass.viveglasssample/files/audio_sample.aac
     * Specs: AAC format, 44.1 kHz, 32 kbps, Stereo.
     */
    override fun startAudioStreaming() {
        glass.startAudioStreaming(
            Microphone.MIC_DIRECTION_SURROUNDING,
            32000,
            16000,
            AudioChannel.MONO,
            audioRecordCallback,
            audioRecordEventCallback
        )
    }

    override fun stopAudioStreaming() {
        glass.stopAudioStreaming()
    }

    /** captureImage()
     * Replacing Simulator Image Samples
     * Path: Transfer your file to the app files directory. Ex:
     * /data/data/com.htc.viveglass.viveglasssample/files/image_sample.heic
     * Specs: HEIC format, 1440x1920 resolution.
     */
    override fun captureImage() {
        val result = glass.captureImage(ImageQuality.DEFAULT)
        _isImageCapturing.value = true
        log.d(TAG, "takePhoto result : $result")
    }

    /** startVideoStreaming()
     * Replacing Simulator Video Samples
     * Path: Transfer your file to the app file directory. Ex:
     * /data/data/com.htc.viveglass.viveglasssample/files/video_sample.mp4
     * Specs: H.264 video (480x856, 30 FPS) with AAC audio (44.1 kHz, 32 kbps, Stereo).
     */
    override fun startVideoStreaming() {
        glass.startVideoStreaming(
            VideoQuality.DEFAULT,
            videoStreamCallback,
            audioStreamCallback,
            streamEventCallback
        )
    }

    override fun stopVideoStreaming() {
        glass.stopVideoStreaming()

    }

    override fun getPreferredLocale(): Locale {
        return glass.userPreferredLocale
    }

    override fun cleanup() {
        streamPlayer?.release()

        mediaPlayer.stop()
        mediaPlayer.release()
    }

    // =======================
    // Release previous resource
    // =======================

    fun releasePreviousPage(routeToRelease: String)
    {
        when(routeToRelease){
            AppDestination.Glasses.route ->{}

            AppDestination.Chat.route ->
            {
                if(_isStartTranscribe.value)
                    stopTranscription()
            }

            AppDestination.Audio.route ->
            {
                if(_isSampleAudioPlaying.value)
                {
                    mediaPlayer.pause()
                    mediaPlayer.seekTo(0)
                    _isSampleAudioPlaying.value = false
                }

                if(_isAudioRecording.value)
                {
                    stopAudioStreaming()
                }
            }

            AppDestination.Camera.route ->
            {
                if(_isVideoStreaming.value)
                    stopVideoStreaming()
            }
        }
    }
    // =======================
    // HEIC to bitmap
    // =======================

    private fun byteToBitmap(data: ByteArray) : Bitmap
    {
        val bitmap =BitmapFactory.decodeByteArray(data,0,data.size)
        val orientation = readHeicExif(data)["Orientation"] ?: return bitmap
        return rotateBitmap(bitmap, orientation)
    }

    private fun readHeicExif(bytes: ByteArray): Map<String, String?> {
        ByteArrayInputStream(bytes).use { input ->
            val exif = ExifInterface(input)

            return mapOf(
                "DateTime" to exif.getAttribute(ExifInterface.TAG_DATETIME),
                "Make" to exif.getAttribute(ExifInterface.TAG_MAKE),
                "Model" to exif.getAttribute(ExifInterface.TAG_MODEL),
                "Orientation" to exif.getAttribute(ExifInterface.TAG_ORIENTATION),
                "GPS Lat" to exif.getAttribute(ExifInterface.TAG_GPS_LATITUDE),
                "GPS Lon" to exif.getAttribute(ExifInterface.TAG_GPS_LONGITUDE)
            )
        }
    }
    private fun rotateBitmap(
        source: Bitmap,
        orientation: String
    ): Bitmap {
        val degrees = when(orientation) {
            ORIENTATION_ROTATE_180.toString() -> 180f
            ORIENTATION_ROTATE_90.toString() -> 90f
            ORIENTATION_ROTATE_270.toString() -> 270f
            ORIENTATION_NORMAL.toString() -> 0f
            else -> 0f
        }

        val matrix = Matrix().apply {
            postRotate(degrees)
        }
        return Bitmap.createBitmap(
            source,
            0,
            0,
            source.width,
            source.height,
            matrix,
            true
        )
    }
}